/*
========================================================================================================================
 Resource API: "recent-visits"

 [GET] "/VeteranAppointmentRequestService/v4/rest/direct-scheduling/site/{site-code}/patient/{assigning-authority}
                /{patient-id}/{eligibility}/visited-in-past-months"

    Parameters:
        assigning-authority (string) The type of patient id to be used
            example: ICN
        patient-id (string|number) The id of patient written in the format of the assigning-authority
        site-code (string|number) The facilities that the patient is registered at
        eligibility (string) direct-eligibility or request-eligibility
    Attributes:
        clinical-service (string|number) Types of Care ID
            example: 323 for Primary Care

========================================================================================================================
 Schema: {
   "hasVisitedInPastMonths" : <boolean>,
   "durationInMonths" : <number>
 }

========================================================================================================================
 Example: {
   "hasVisitedInPastMonths" : false,
   "durationInMonths" : 12
 }

========================================================================================================================
 */
define([
    'underscore',
    'models/abstract-model',
    'User',
], function(_, Model, user) {
    'use strict';


    var DIRECT = 'direct-eligibility';
    var REQUEST = 'request-eligibility';
    var NO_LIMIT_REQUIRED = 0;


    var LimitModel = Model.extend({
        resourceName: 'recent-visits',

        eligibilityType: DIRECT,

        /**
         * Makes the model use the direct eligibility resource
         * @default
         * @return {void}
         */
        setAsDirectModel: function() {
            this.eligibilityType = DIRECT;
        },

        /**
         * Makes the model use the request eligibility resource
         * @return {void}
         */
        setAsRequestModel: function() {
            this.eligibilityType = REQUEST;
        },

        /**
         * @param {String|Number} siteCode The facilities that the patient is registered at
         * @return {String}
         */
        createUrl: function(siteCode) {
            var resource = this.getResource();

            return resource.formatHref({
                'eligibility': this.eligibilityType,
                'site-code': siteCode,
                'assigning-authority': user.get('idType'),
                'patient-id': user.id,
            });
        },

        /**
         * @param {String|Number} siteCode The facilities that the patient is registered at
         * @param {String} institutionCode
         * @param {String|Number} clinicalService Types of Care IDs
         * @param {*} [options] Fetch options
         * @return {JQueryXHR}
         */
        fetchRecentVisits: function(siteCode, institutionCode, clinicalService, options) {
            var request = _.extend({}, options);

            request.url = this.createUrl(siteCode);
            request.data = _.extend({}, request.data, {
                'clinical-service': clinicalService,
                'institution-code': institutionCode,
            });

            return this.fetch(request);
        },

        hasRecentVisit: function() {
            var duration = this.get('durationInMonths');
            if (duration === NO_LIMIT_REQUIRED) {
                return true;
            }
            return Boolean(this.get('hasVisitedInPastMonths'));
        },
    });


    LimitModel.DIRECT = DIRECT;
    LimitModel.REQUEST = REQUEST;


    return LimitModel;
});
